#!/usr/bin/perl

#---------------------------------------------------------------------
# Copyright (C) 2003 Chad Redman <chad_eu2@katica.org>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#---------------------------------------------------------------------


# File:     EventParser.pl
# Version:  0.9.1
# Author:   Chad Redman <chad_eu2@katica.org>

# $Id: EventParser.pl,v 1.3 2003/03/14 04:03:36 chad Exp $


package Symbol;

sub new {
    my ($pkg, $text, $line, $character) = @_;
    my $self = [$text, $line, $character];
    return bless $self, $pkg;
}

sub text {
    my $self = shift;
    return $self->[0];
}

sub line {
    my $self = shift;
    return $self->[1];
}

sub char {
    my $self = shift;
    return $self->[2];
}

sub location {
    my $self = shift;
    return (@$self[1,2]);
}

package EventParser;

$EventParser::VERSION = '0.9.1';

%db_countries = (ADE=>1, AFG=>1, AKK=>1, ALB=>1, ALD=>1, AMI=>1, ANN=>1, APA=>1, AQU=>1, ARA=>1, ARG=>1, ARK=>1, ARM=>1, ASH=>1, ASS=>1,
        AST=>1, ATH=>1, ATJ=>1, AUV=>1, AYU=>1, AZT=>1, BAD=>1, BAL=>1, BAN=>1, BAY=>1, BEI=>1, BEN=>1, BGL=>1, BOH=>1, BOS=>1,
        BOU=>1, BRA=>1, BRE=>1, BRI=>1, BRZ=>1, BUL=>1, BUR=>1, BYZ=>1, CAL=>1, CAM=>1, CAN=>1, CAS=>1, CAT=>1, CHA=>1, CHE=>1,
        CHG=>1, CHI=>1, CHM=>1, COL=>1, COR=>1, CRE=>1, CRI=>1, CRO=>1, CYP=>1, CYR=>1, DAH=>1, DAI=>1, DAK=>1, DAN=>1, DEL=>1,
        DLH=>1, DUL=>1, EIR=>1, ENG=>1, ETH=>1, FEZ=>1, FIN=>1, FLA=>1, FPR=>1, FRA=>1, FRI=>1, FUC=>1, GEL=>1, GEN=>1, GEO=>1,
        GRA=>1, GRE=>1, GUJ=>1, HAB=>1, HAI=>1, HAN=>1, HAU=>1, HEI=>1, HEL=>1, HES=>1, HOL=>1, HSA=>1, HUN=>1, HUR=>1, HYD=>1,
        INC=>1, IRA=>1, ITA=>1, JAI=>1, JER=>1, KAL=>1, KAR=>1, KAZ=>1, KHM=>1, KLE=>1, KNI=>1, KOL=>1, KON=>1, KOR=>1, KSH=>1,
        KUR=>1, KZK=>1, LAP=>1, LAT=>1, LIT=>1, LOR=>1, LUA=>1, LUX=>1, MAA=>1, MAG=>1, MAH=>1, MAI=>1, MAL=>1, MAM=>1, MAN=>1,
        MCH=>1, MEC=>1, MER=>1, MEX=>1, MIN=>1, MKS=>1, MLC=>1, MLO=>1, MLW=>1, MOG=>1, MOL=>1, MOR=>1, MOS=>1, MTR=>1, MUN=>1,
        MUS=>1, MYA=>1, MYS=>1, NAP=>1, NAT=>1, NAV=>1, NIP=>1, NOG=>1, NOR=>1, NVG=>1, OHI=>1, OLD=>1, OMA=>1, ORI=>1, ORL=>1,
        PAP=>1, PAR=>1, PEG=>1, PER=>1, PFA=>1, PIR=>1, PIS=>1, POL=>1, POM=>1, POR=>1, PRM=>1, PRO=>1, PRU=>1, PSK=>1, PUR=>1,
        QAR=>1, QUE=>1, RAG=>1, REB=>1, ROY=>1, RUS=>1, RYA=>1, SAC=>1, SAR=>1, SAV=>1, SCO=>1, SER=>1, SHA=>1, SHL=>1, SIB=>1,
        SIC=>1, SIE=>1, SLZ=>1, SON=>1, SPA=>1, SPR=>1, STE=>1, STR=>1, SUD=>1, SUZ=>1, SWE=>1, TAU=>1, TEK=>1, THU=>1, TIB=>1,
        TIM=>1, TOS=>1, TRE=>1, TRI=>1, TUN=>1, TUR=>1, TVE=>1, U00=>1, U01=>1, U02=>1, U03=>1, U04=>1, U05=>1, U06=>1, U07=>1,
        U08=>1, U09=>1, U10=>1, U11=>1, U12=>1, U13=>1, U14=>1, U15=>1, U16=>1, U17=>1, U18=>1, U19=>1, UKR=>1, USA=>1, UZB=>1,
        VEN=>1, VIE=>1, VIJ=>1, WAL=>1, WLS=>1, WUR=>1, XHO=>1, ZAN=>1, ZAP=>1, ZIM=>1, ZUL=>1,);


%db_cultures = (abenaki=>1, aborigin=>1, afghani=>1, aka=>1, albanian=>1, aleutian=>1, altai=>1, amazonian=>1, andean=>1,
    anglosaxon=>1, arabic=>1, armenian=>1, ashanti=>1, aztek=>1, baltic=>1, baluchi=>1, bantu=>1, basque=>1,
    bengali=>1, berber=>1, burmanese=>1, canary=>1, cantonese=>1, caribbean=>1, cherokee=>1, cree=>1, creek=>1,
    czech=>1, dakota=>1, delaware=>1, dravidian=>1, dutch=>1, dyola=>1, ethiopian=>1, filippine=>1, french=>1,
    gaelic=>1, georgian=>1, german=>1, greek=>1, guajiro=>1, gujarati=>1, han=>1, hawaiian=>1, hindi=>1, huron=>1,
    iberian=>1, indonesian=>1, inuit=>1, iroquis=>1, italian=>1, japanese=>1, javan=>1, khazak=>1, khmer=>1,
    kongolese=>1, korean=>1, kurdish=>1, laotian=>1, lithuanian=>1, madagasque=>1, magyar=>1, malay=>1, mali=>1,
    maltese=>1, manchu=>1, marathi=>1, mataco=>1, mayan=>1, melanese=>1, mesoamerican=>1, mississippian=>1,
    mongol=>1, naskapi=>1, navajo=>1, none=>1, nubian=>1, patagonian=>1, persian=>1, polish=>1, polynese=>1,
    romanian=>1, russian=>1, ruthenian=>1, scandinavian=>1, senegambian=>1, shawnee=>1, shona=>1, sikh=>1,
    slavonic=>1, slovak=>1, somali=>1, swahili=>1, swiss=>1, syrian=>1, teremembe=>1, thai=>1, tibetan=>1,
    tuareg=>1, tupinamba=>1, turkish=>1, ugric=>1, uzbehk=>1, vietnamese=>1, yorumba=>1, zapotek=>1,);

%db_religions = (catholic=>1, counterreform=>1, protestant=>1, reformed=>1, orthodox=>1, sunni=>1, shiite=>1,
    confucianism=>1, buddhism=>1, hinduism=>1, pagan=>1);

%db_continents = (europe=>1, america=>1, asia=>1, africa=>1, oceania=>1,);

%db_techs = (latin=>1, torthodox=>1, muslim=>1, china=>1, exotic=>1,);

%db_diplo =  (aristocracy=>1, centralization=>1, innovative=>1, mercantilism=>1, land=>1, offensive=>1, quality=>1, serfdom=>1, );

%db_buildings = (shipyard=>1, barrack=>1, bailiff=>1, courthouse=>1, cityrights=>1, );

## Note: -1 means a random manufactury
%db_manufacturies = ( navalequipment=>1, luxury=>1, goods=>1, refinery=>1, weapons=>1, '-1'=>1, );



### Dates use text for month. This maps each to a number
%db_months = qw(
    january    01
    february   02
    march      03
    april      04
    may        05
    june       06
    july       07
    august     08
    september  09
    october    10
    november   11
    december   12
);

sub LogLevel {
    $EventParser::loglevel = shift;
}

sub log {
    my ($level, $text) = @_;

    if ($level <= $EventParser::loglevel) {
        print STDOUT ( ($EventParser::escapeHTML) ? CGI::escapeHTML("$text\n") : "$text\n" );
    }
}

sub new {
    my ($pkg) = @_;
    my $self = {
        'symbols' => [],
        'ids' => [],
    };

    return bless $self, $pkg;
}

sub fileHandle {
    my $self = shift;
    if (@_) { $self->{fileHandle} = shift }
    return $self->{fileHandle};
}

sub source {
    my $self = shift;
    if (@_) { $self->{source} = shift }
    return $self->{source};
}


sub addSymbol {
    my ($self, $sym) = @_;
    push @{$self->{symbols}}, $sym;
}


sub _tokenize {
    my $self = shift;

    my @lines = ();

    if ($self->fileHandle) {
        my $fh = $self->fileHandle;
        @lines = <$fh>;
        close $fh;
    } else {
        @lines = split(/\r?\n/, $self->source);
    }

    if (!@lines) {
        EventParser::log 0, "Error: No input for Event Parser!\n";
        return 0;
    }

    my @tokens = ();
    my $line;
    my $line_no = 0;
    my $char_no = 0;
    while (@lines) {
        $line = splice(@lines,0,1);
        ++$line_no;
        $char_no = 0;
        while (length($line)) {
            if ($line =~ /^\s+/) { $char_no += length($&); $line = $'; next;}        # whitespace
            if ($line =~ /^\#.*/) { $char_no += length($&); $line = $'; next;}        #comment

            if ($line =~ /^(^\".*\")\s*$/) { $self->addSymbol(new Symbol($1, $line_no, $char_no)); $char_no += length($&); $line = $'; next;}
                #quoted string #1. Matches "Zweite Armeecorps "Neumark" " (nested double-quotes, can only parse by assuming it takes up a single line)

            if ($line =~ /^(\"[^\"]*\")/) { $self->addSymbol(new Symbol($1, $line_no, $char_no)); $char_no += length($&); $line = $'; next;}
                #quoted string #2. basic--not nesting

            if ($line =~ /^([0-9+-.]+)/) { $self->addSymbol(new Symbol($1, $line_no, $char_no)); $char_no += length($&); $line = $'; next;}
                #number
            if ($line =~ /^([A-Za-z0-9_.]+|=|\{|\})/) { $self->addSymbol(new Symbol($1, $line_no, $char_no)); $char_no += length($&); $line = $'; next;}
                #any other valid token
            else {
                EventParser::log 0, "Warning: unlikely token at line $line_no char $char_no : '$line'";
                if ($line =~ /^(\S+)/) { $self->addSymbol(new Symbol($1, $line_no, $char_no)); $char_no += length($&); $line = $'; next;}
                ## The next non-whitespace string of characters. Not a valid token, but save it anyway to keep the flow going
            }
        }
    }
}

sub parse {
    my $self = shift;
    $self->_tokenize;

    my $toks = $self->{symbols};

    while (&matches($toks, qw( event = { ))) {
        my $event = Event->new;
        $event->parse($toks) or return _abortParseError($toks);
        #$self->addEvent( $event );
        push @{$self->{ids}}, $event->{id};
    }

    if (scalar @$toks == 0) {  ## nothing left in the whole file
        return 1;
    } else {
        EventParser::log 0, "Warning: Expected 'event = {' at line " . $toks->[0]->line . " char " . $toks->[0]->char;
        EventParser::warnBadState(ref $obj, '-TOP-', $toks);

        _abortParseError($toks)
    }
}


#  ------- UTILITIES ---------
sub matches {            # (\@tokens, @fields)
    my ($toks, @fields) = @_;

    foreach (0 .. $#fields) {
        return 0 if !defined $toks->[$_];

        my $match = $fields[$_];
        my $src = $toks->[$_]->text;

        ## A number can be 0, +/- 1...N, or +/- "inf". A "v" after a number indicates to ignore if the lenient option is on
        if ($match =~ /^:int,(-?[0-9]+|-inf)(v?),(-?[0-9]+|inf)(v?):$/) {
            my ($lo, $lo_v, $hi, $hi_v) = ($1, $2, $3, $4);

            return 0 if $src !~ /^(-?[0-9]+)$/;

            if ($EventParser::opt_check_range ne 'none') {
                if (($lo ne '-inf') && ($src < $lo ) && !($lo_v && $EventParser::opt_check_range eq 'loose')) {
                    EventParser::log 0, "Integer $src out of range ($lo to $hi)";
                    return 0;
                }

                if (($hi ne 'inf') && ($src > $hi ) && !($hi_v && $EventParser::opt_check_range eq 'loose')) {
                    EventParser::log 0, "Integer $src out of range ($lo to $hi)";
                    return 0;
                }
            }
        } elsif ($match =~ /^:province(,rand)?:$/) {
            next if (($1 eq ',rand') and ($src <= -1 and $src >= -4));
            return 0 if $src !~ /^[0-9]+$/;
            if ($src < 1 or $src > 1614) {
                EventParser::log 0, "Integer $src out of range ($1 to $2)";
                return 0;
            }
        } elsif ($match =~ /^:ctag(,-1)?:$/) {
            next if (($1 eq ',-1') and ($src == -1 or $src == -3));
            return 0 if $src !~ /^[A-Za-z0-9]{3}$/;
            if (!defined $EventParser::db_countries{uc $src}) {
                EventParser::log 0, "Country tag '$src' invalid";
                return 0;
            }
        } elsif ($match =~ /^:culture(,-1)?:$/) {
            next if (($1 eq ',-1') and ($src == -1));
            if (!defined $EventParser::db_cultures{lc $src}) {
                EventParser::log 0, "Culture '$src' invalid";
                return 0;
            }
        } elsif ($match eq ':religion:') {
            if (!defined $EventParser::db_religions{lc $src}) {
                EventParser::log 0, "Religion '$src' invalid";
                return 0;
            }
        } elsif ($match eq ':continent:') {
            if (!defined $EventParser::db_continents{lc $src}) {
                EventParser::log 0, "Continent '$src' invalid";
                return 0;
            }
        } elsif ($match eq ':tech:') {
            if (!defined $EventParser::db_techs{lc $src}) {
                EventParser::log 0, "Tech group '$src' invalid";
                return 0;
            }
        } elsif ($match eq ':diplo:') {
            if (!defined $EventParser::db_diplo{lc $src}) {
                EventParser::log 0, "Diplomacy category '$src' invalid";
                return 0;
            }
        } elsif ($match eq ':building:') {
            if (!defined $EventParser::db_buildings{lc $src}) {
                EventParser::log 0, "Building type '$src' invalid";
                return 0;
            }
        } elsif ($match eq ':manufactury:') {
            if (!defined $EventParser::db_manufacturies{lc $src}) {
                EventParser::log 0, "Manufactury type '$src' invalid";
                return 0;
            }
        } elsif ($match eq ':year:') {
            return 0 if $src !~ /^-?[0-9]+$/;
            if (($EventParser::opt_range_check eq 'strict') && ($src < 1419 or $src > 1821)) {
                EventParser::log 0, "Warning: Integer $src out of range ($1 to $2)";
                return 0;
            }
        } elsif ($match eq ':month:') {
            if (!defined $EventParser::db_months{lc $src}) {
                EventParser::log 0, "Month '$src' invalid";
                return 0;
            }
        } else {
            return 0 if $src !~ /^$match$/ and lc($src) !~ /^$match$/;
        }
   }

   1;
}
sub balanceParen {
    my ($toks, $pre, $post) = @_;
    my $count = 1;

    while (@$toks) {
        my $cur = (shift @$toks)->text;
        if ($cur eq $pre) { ++$count; }
        elsif ($cur eq $post) { --$count; return if $count == 0; }
    }

    EventParser::log 0, "Unmatched '$pre' before end of buffer after '" . join(' ', @$toks[0..10]) . "'";
    return;
}


sub warnBadState {          #($state, $buf);
    my ($class, $state, $toks) = @_;
    my @jointags;
    foreach ( 0 .. 10) {
        push @jointags, ($toks->[$_]->text) if defined $toks->[$_];
    }

    EventParser::log 0, "Stuck in object $class, state $state, at line "
                   . $toks->[0]->line . " char "
                   . $toks->[0]->char . ", with buffer '" . join(' ', @jointags) . "'";
}

sub _abortParseError {
    my ($tokens) = @_;
    #EventParser->warnBadState('??', $tokens);

    #my $time = time();
    #open (ERRLOG, "> err_$time.log");       ## just ignore errors, instead of exiting
    #print ERRLOG join(' ', @$tokens) . "\n";
    #close ERRLOG;
    return 0;
}




#-------------------------------------
package Event;

sub new {
    my $pkg = shift;
    my $self = {
        'id' => undef,
        #....
    };

    return bless $self, $pkg;
}


sub parse {
    my ($obj, $toks) = @_;
    my $state = '<start>';
    my @state_stack = ();

    while ($state ne '<end>') {
        if (scalar @$toks == 0) {  ## nothing left in the whole file
            $state = '<end>';
            next;
        }

        if ($state eq '<start>') {
            if ( EventParser::matches( $toks, qw(event = { ))) {
                $state = '<event>';
                splice (@$toks, 0, 3);
            } else {
                EventParser::warnBadState(ref $obj, $state, $toks);
                EventParser::balanceParen( $toks, '{', '}');   ## This particular error handler may need refinement
                $state = '<end>';
            }
        }

        elsif ($state eq '<event>') {
            # id
            if ( EventParser::matches( $toks, qw( id = :int,0,4294967296: ))) {
                $obj->{'id'} = $toks->[2]->text;
                splice (@$toks, 0, 3);
                next;
            }

            # random
            if ( EventParser::matches( $toks, qw( random = (yes|no) ))) {
                splice (@$toks, 0, 3);
                next;
            }

            # country tag
            if ( EventParser::matches( $toks, qw( country = :ctag: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            # province tag. Reported to only work since 1.06. '-1' is invalid
            if ( EventParser::matches( $toks, qw( province = :province: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            # name, desc
            if ( EventParser::matches( $toks, qw( (name|desc) = .* ))) {
                splice (@$toks, 0, 3);
                next;
            }

            # style
            if ( EventParser::matches( $toks, qw( style = :int,0,5: ))) {
                splice (@$toks, 0, 3);
                next;
            }


            # date, deathdate
            if ( EventParser::matches( $toks, qw( (date|deathdate) = { day = :int,0,31: month = :month: year = :year: } ))) {
                splice (@$toks, 0, 13);
                next;
            }

            # date, deathdate #2 - different order
            if ( EventParser::matches( $toks, qw( (date|deathdate) = { year = :year: month = :month: day = :int,0,31: } ))) {
                splice (@$toks, 0, 13);
                next;
            }

            # date, deathdate #2 -- just year
            if ( EventParser::matches( $toks, qw( (date|deathdate) = { year = :year: } ))) {
                splice (@$toks, 0, 7);
                next;
            }

            # offset
            if ( EventParser::matches( $toks, qw( offset = :int,0,inf: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            # trigger block
            if ( EventParser::matches( $toks, qw( trigger = { ))) {
                $state = '<trigger>';
                splice (@$toks, 0, 3);
                next;
            }

            # action block
            if ( EventParser::matches( $toks, qw( action_[abcd] = { ))) {
                $state = '<action>';
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( } ))) {
                $state = '<end>';
                splice (@$toks, 0, 1);
                next;
            }
            #....................
            do {
                EventParser::warnBadState(ref $obj, $state, $toks);
                EventParser::balanceParen( $toks, '{', '}');
                $state = '<end>';
            }

        }
        elsif ($state eq '<trigger>') {
            if ( EventParser::matches( $toks, qw( } ))) {
                #$state = '<event>';
                $state = ((pop @state_stack) || '<event>');
                splice (@$toks, 0, 1);
                next;
            }

            ## Note: Doesn't check for 2 or more triggers for AND/OR
            if ( EventParser::matches( $toks, qw( (and|or|not) = { ))) {
                push @state_stack, $state;
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( event = :int,0,4294967296: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( year = :year: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( (alliance|dynastic|vassal|war) = { country = :ctag: country = :ctag: } ))) {
                splice (@$toks, 0, 10);
                next;
            }

            if ( EventParser::matches( $toks, qw( atwar = (yes|no) ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( provinceculture = { prov = :province: data = :culture: } ))) {
                splice (@$toks, 0, 10);
                next;
            }

            if ( EventParser::matches( $toks, qw( provincereligion = { province = :province: data = :religion: } ))) {
                splice (@$toks, 0, 10);
                next;
            }

            if ( EventParser::matches( $toks, qw( religion = :religion: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( (leader|monarch) = :int,0,19999: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( (badboy|countrysize) = :int,0,inf: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( continent = :continent: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            ## style 1 of control/owned trigger: missing data means the country defined for the event
            if ( EventParser::matches( $toks, qw( (control|owned) = { province = :province: } ))) {
                splice (@$toks, 0, 7);
                next;
            }

            ## style 2 of control/owned trigger
            if ( EventParser::matches( $toks, qw( (control|owned) = { province = :province: data = :ctag,-1: } ))) {
                splice (@$toks, 0, 10);
                next;
            }

            if ( EventParser::matches( $toks, qw( core = { province = :province: data = :ctag,-1: } ))) {
                #The core trigger is fixed in 1.06
                #EventParser::log 0, sprintf("Note: The core trigger is reputed to not work (line %d char %d)", $toks->[0]->location);
                splice (@$toks, 0, 10);
                next;
            }

            if ( EventParser::matches( $toks, qw( (cot|discovered) = :province: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( (exists|neighbour) = :ctag: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( relation = { country = :ctag: value = :int,-200,200: } ))) {
                splice (@$toks, 0, 10);
                next;
            }

            if ( EventParser::matches( $toks, qw( domestic = { type = :diplo: value = :int,0,10: } ))) {
                splice (@$toks, 0, 10);
                next;
            }

            if ( EventParser::matches( $toks, qw( (infra|trade) = :int,0,10: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( (land|naval) = :int,0,60: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( stability = :int,-15v,15v: ))) {
                splice (@$toks, 0, 3);
                next;
            }

            ## New for 1.06
            if ( EventParser::matches( $toks, qw( ai = (yes|no) ))) {
                splice (@$toks, 0, 3);
                next;
            }

            ## New for 1.06
            if ( EventParser::matches( $toks, qw( flag = .* ))) {
                splice (@$toks, 0, 3);
                next;
            }

            do {
                EventParser::warnBadState(ref $obj, $state, $toks);
                EventParser::balanceParen( $toks, '{', '}');
                #$state = '<event>';
                $state = ((pop @state_stack) || '<event>');
            }
        }

        elsif ($state eq '<action>') {
            if ( EventParser::matches( $toks, qw( } ))) {
                $state = '<event>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( EventParser::matches( $toks, qw( name = .* ))) {
                splice (@$toks, 0, 3);
                next;
            }

            if ( EventParser::matches( $toks, qw( command = { ))) {
                $state = '<action_command>';
                splice (@$toks, 0, 3);
                next;
            }

            do {
                EventParser::warnBadState(ref $obj, $state, $toks);
                EventParser::balanceParen( $toks, '{', '}');
                $state = '<event>';

            }
        }

        elsif ($state eq '<action_command>') {
            if ( EventParser::matches( $toks, qw( } ))) {
                $state = '<action>';
                splice (@$toks, 0, 1);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (alliance|breakdynastic|breakvassal|dynastic|independence|inherit|vassal|war) which = :ctag,-1: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = casusbelli which = :ctag,-1: value = :int,0,inf: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = relation which = :ctag,-1: value = :int,-400,400: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (conversion|heretic|religiousrevolt) which = :province,rand: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = provincereligion which = :province,rand: value = :religion: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = religion which = :religion: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (add_countryculture|remove_countryculture) which = :culture: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = provinceculture which = :province,rand: value = :culture,-1: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (conquistador|explorer) which = :province,rand: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (leader|monarch|sleepleader|sleepmonarch|wakeleader|wakemonarch) which = :int,0,19999: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (adm|mil|dip) which = :int,-15,15: value = :int,0,inf: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = country which = :ctag: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = technology which = :tech: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (capital|addcore|removecore|cot) which = :province: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = secedeprovince which = :ctag: value = :province: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = province_revoltrisk which = :province: value = :int,-100,100: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = population which = :province,rand: value = :int,-inf,inf: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = provincetax which = :province,rand: value = :int,-10v,10v: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = manpower value = :int,-100v,100v: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = provincemanpower which = :province,rand: value = :int,-10v,10v: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = natives which = :province: value = :int,0,9: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = mine which = :province,rand: value = :int,-200v,200v: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = domestic which = :diplo: value = :int,-10,10: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (infra|trade|land|naval|treasury|cash|inflation|vp) value = :int,-inf,inf: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = loansize which = :int,1,inf: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = stability value = :int,-15v,15v: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (revolt|colonialrevolt) which = :province,rand: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = revoltrisk which = :province,rand: value = :int,-100,100: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            #revoltrisk #2 - missing months defaults to 12
            if ( EventParser::matches( $toks, qw( type = revoltrisk value = :int,-100,100: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = fortress which = :province,rand: value = :int,-6,6: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (gainbuilding|losebuilding) which = :province,rand: value = :building: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = gainmanufactory which = :province,rand: value = :manufactury: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = losemanufactory which = :province,rand: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (diplomats|colonists|merchants|missionaries) value = :int,-6,6: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (desertion|inf|cav|art|warships|galleys|transports|infantry|cavalry|artillery) which = :province,rand: value = :int,0,inf: ))) {
                splice (@$toks, 0, 9);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = flag which = :int,0,6: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            if ( EventParser::matches( $toks, qw( type = (trigger|sleepevent) which = :int,0,4294967296: ))) {
                splice (@$toks, 0, 6);
                next;
            }

            ## New in 1.06
            if ( EventParser::matches( $toks, qw( type = (setflag|clrflag|ai) which = .* ))) {
                splice (@$toks, 0, 6);
                next;
            }


            do {
                EventParser::warnBadState(ref $obj, $state, $toks);
                EventParser::balanceParen( $toks, '{', '}');
                $state = '<action>';
            }
        }

        else {
                EventParser::warnBadState(ref $obj, $state, $toks);
                EventParser::balanceParen( $toks, '{', '}');
                $state = '<end>';

        }
    } #while

    1;

}



#-------------------------------------


package main;

use FileHandle;
use Getopt::Long;

GetOptions('check-range=s', 'nowarn-dup-ids', 'full-id-report') or exit &Syntax();

$EventParser::opt_check_range = $opt_check_range ||  'strict';

foreach my $arg (@ARGV) {
    if (-d $arg) {
        if (opendir DIR, $arg) {
            #push @filelist, sort(grep /.*\.txt$/, map "$arg/$_", readdir DIR);
            push @filelist, sort(grep -T, map "$arg/$_", readdir DIR);           ## include all text files
            closedir DIR;
        } else {
            EventParser::log 0, "Error opening directory $arg";
        }
    } else {
        push @filelist, $arg;
    }
}

my %ids;

foreach $file (@filelist) {
    my $fh = new FileHandle("< $file");
    #next if $file eq 'eventlist.txt';
    if (defined $fh) {
        my $parser = new EventParser();
        $parser->fileHandle($fh);
        print "    **** $file\n";
        $parser->parse;

        foreach my $id (@{$parser->{ids}}) {
            if (defined $ids{$id}) {
                $opt_nowarn_dup_ids or EventParser::log 0, "Duplicate Event ID $id, first defined in $ids{$id}";
            } else {
                $ids{$id} = $file;
            }
        }
    } else {
        print "!!! Error opening file $file\n";
        next;
    }
}

if ($opt_full_id_report) {
    my $first = '';
    my $last = '';
    foreach my $id (sort { $a <=>$b } keys %ids) {
        if ($first eq '') {
            $first = $id;
        }

        if ($last eq '') {
            $last = $id;
        } elsif ($id == $last+1) {
            $last = $id;
        } else {
            push @idranges, (($first == $last) ? $first : "$first-$last");
            $first = $id;
            $last = '';
        }
    }

    push @idranges, (($first == $last) ? $first : "$first-$last");

    EventParser::log 0, "Event ID ranges: " . join(',', @idranges) . "\n";


}


sub Syntax {
    print STDERR "Syntax: $0
        [--check-range=strict|loose|none]
        [--nowarn-dup-ids]
        [--full-id-report] filename|directory filename|directory ...\n";
    print STDERR "    --check-range: For some of the event directives that require
            numbers, the range of valid values is open-ended. A reasonable range
            is defined in this program, based on common usage. If you would like
            to turn off warnings for these open-ended values, but keep warnings
            related to ranges that are hard-coded in the game (such as ferocity of 0-9),
            choose the loose option. If you don't want any out-of-range errors, choose
            the none option. The default is strict.\n";
    print STDERR "    --nowarn-dup-ids: don't give warnings when an event ID is reused
            in any of the files parsed so far.\n";
    print STDERR "    --full-id-report: At the end of processing, report the complete set of
            event ID ranges that were obtained from all files parsed. This gives output
            in the format of '100-120,123,125-150', etc.\n";
    exit 1;
}



exit 0;
