#---------------------------------------------------------------------
# Copyright (C) 2003 Chad Redman <chad_eu2@katica.org>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#---------------------------------------------------------------------

package Savefile;

# File-scoped variables
use vars '%MONTHS', '@FLAGS', '@DISCOVERIES', '@ESTABLISHMENTS', '$dbData', '%POLICY_FIELDS', '%TECH_FIELDS', '@watchesGlobal', '@watchesCountry', '@watchesCoT', '$outputDir', '@CONTINENTS', '@REGIONS', '@AREAS', '$loglevel';

$Savefile::revision = '$Id: Savefile.pm,v 1.2 2003/02/12 03:38:14 chad Exp $';
$Savefile::VERSION='1.0';


my $loglevel = 1;

### Savefile dates store the month as text. This maps each to a number
my %MONTHS = qw(
    january    01
    february   02
    march      03
    april      04
    may        05
    june       06
    july       07
    august     08
    september  09
    october    10
    november   11
    december   12
);


### In the globaldata section, the flags = { num(s) } indicates special game rules in effect. Numbers are from 0-6
my @FLAGS = (
        'Tortesillas',  #Set if the Treaty of Tortesillas has happened
        'Reformation',  #Set if the Reformation has happened. No provinces will be protestant unless this is set.
        'Calvin',       #Set if Jean Calvin has happened. No provinces will be reformed unless this is set.
        'Council of Trent',    #Set if the Council of Trent has happened
        'Edict of Tolerance',  #Set if the Edict of Tolerance has happened
        'French Tricolor',     #If set, the French flag is changed to the Tricolor
        'English Union Jack',  #If set, the English flag is changed to the Union Jack
        );


### The discoveries section of globaldata gives yes/no for each of these
my @DISCOVERIES = qw(
        america
        asia
        quebec
        greatlake
        capehorn
        goodhope
        hudsonbay
        alaska
        panama
        inlandchina
        amazones
        northam
        india
        kamtchatska
        california
        pacific
        australia
);


### The establishments section of globaldata gives yes/no for each of these
my @ESTABLISHMENTS = qw(
        colonyamerica
        colonyafrica
        colonysiberia
        colonyoceania
        tpindia
        tpchina
        tpjapan
);


@CONTINENTS = ();   ## Created after reading province.csv
@REGIONS = ();   ## Created after reading province.csv
@AREAS = ();   ## Created after reading province.csv



## maps the diplomatic policy name to an array position
my %POLICY_FIELDS = qw(
    aristocracy       0
    centralization    1
    innovative        2
    mercantilism      3
    offensive         4
    land              5
    quality           6
    serfdom           7
);


## maps the tech area name to an array position
my %TECH_FIELDS = qw(
    land              0
    naval             1
    stability         2
    trade             3
    infra             4
);

## not used yet.
#my %INCOME_FIELDS = qw(
#);

#my %EXPENSE_FIELDS = qw(
#);

#my %BASEVP_FIELDS = qw(
#);



$dbData = {
        'provinces' => {},
        'countries' => {},
        };

my @EU2TREES = ();


sub LogLevel {
    $Savefile::loglevel = shift;
}

sub log {
    my ($level, $text) = @_;

    if ($level <= $::loglevel) {
        print STDOUT "$text\n";
        open (LOG, ">> $Savefile::outputDir/daemon_log.txt");
        print LOG "$text\n";
        close LOG;
    }
}


sub getDbProvince {
    my $id = shift;
    return $dbData->{'provinces'}->{$id};
}

sub getDbCountry {
    my $tag = shift;
    return $dbData->{'countries'}->{$tag};
}

sub EU2Tree {
    my (@dirpath) = @_;
    if (@dirpath) {
        push @EU2TREES, @dirpath;
    } else {
        return @EU2TREES;
    }
}


sub initDbData {
    &_initProvinces;
    &_initCountries;
    &_initCountryNames;

}


sub _locateFile {
    my ($subpath) = @_;
    foreach my $path (@EU2TREES) {
        if (-r "$path/$subpath") {
            return "$path/$subpath";
        }
    }

    die "Could not find $subpath within directory trees [\n\t'" . join("',\n\t'", @EU2TREES) . "']\n";
}


sub _initProvinces {
    my $file = &_locateFile('Db/province.csv') or return 0;

    if (!open (PROV, "< $file")) {
        Savefile::log 0, "Could not open \"$file\"";
        return 0;
    }

    $_ = <PROV>;  #skip first line
    while (<PROV>) {
        my @fields = split(";", $_);

        next if $fields[13] =~ /^[568]$/;    # ignore water provinces (5=Ocean, 6=River,  8=Terra incognita[unused?])

        my $prov = new Savefile::Province;

        $prov->{'id'} = $fields[0];
        $prov->{'name'} = $fields[1];
        $prov->{'religion'} = $fields[3];
        $prov->{'culture'} = $fields[4];
        $prov->{'manpower'} = $fields[11];
        $prov->{'income'} = $fields[12];
        $prov->{'terrain'} = $fields[13];
        $prov->{'mine'} = $fields[15];    # minevalue, right? Same as 'mine' in the scenarios?
        $prov->{'goods'} = $fields[16];
        $prov->{'natives'} = $fields[20]; #Native Combat Strength (div 10)- Here it is native strength. Is it the same in the scenario?
        $prov->{'area'} = $fields[46];
        $prov->{'region'} = $fields[47];
        $prov->{'continent'} = $fields[48];

        #Not defined here: whiteman, tax

        $dbData->{'provinces'}->{ $fields[0] } = $prov;
    }

    close PROV;

    my %conts;
    my %area;
    my %region;
    my $prov;
    foreach $prov (values %{$dbData->{'provinces'}}) {
        ++$conts{ $prov->{continent} };
        ++$area{ $prov->{area} };
        ++$region{ $prov->{region} };
    }

    @CONTINENTS = sort keys %conts;
    @REGIONS = sort keys %region;
    @AREAS = sort keys %area;
}


# Note 1: Tag END isn't really for a country, just the end of file signal. It does have latin tech, though!
# Note 2: Country tag SPR isn't in text.csv, so it's probably just cruft. We could filter it out if it's a nuisance
sub _initCountries {
    my $file = &_locateFile('Db/country.csv') or return 0;

    if (!open (CTRY, "< $file")) {
        Savefile::log 0, "Could not open \"$file\"";
        return 0;
    }

    $_ = <CTRY>;  #skip header fields on first line
    while (<CTRY>) {
        my @fields = split(";", $_);

        next if ($fields[0] =~ /^END$/);
        my $ctry = new Savefile::Country;

        $ctry->{'name'} = '<Not in Config/text.csv!>';
        $ctry->{'tag'} = $fields[0];
        $ctry->{'tech_group'} = $fields[3];
        $ctry->{'policies'} = [ @fields[8..15] ];
        $ctry->{'policystring'} = join('-', @fields[8..15]);

        $ctry->{'electors'} = $fields[16];

        $dbData->{'countries'}->{ $fields[0] } = $ctry;
    }

    close CTRY;
}

sub _initCountryNames {
    my $file = &_locateFile('Config/text.csv') or return 0;

    if (!open (TEXT, "< $file")) {
        Savefile::log 0, "Could not open \"$file\"";
        return 0;
    }

    $_ = <TEXT>;  #skip first line
    while (<TEXT>) {
        my @fields = split(";", $_);
        my $key = $fields[0];
        if (defined $dbData->{'countries'}->{$key}) {
            $dbData->{'countries'}->{$key}->{'name'} = $fields[1];
        }
    }

    close TEXT;
}


sub new {
    my ($pkg, $filename) = @_;
    my $self = {
        'filename' => $filename,
        'size' => undef,
        'timestamp' => undef,
        'header' => undef,
        'globaldata' => undef,
        'provinces' => {},
        'cots' => {},
        'countries' => {},
    };

    $self->{'timestamp'} = (stat $filename)[9];
    $self->{'size'} = (stat $filename)[7];

    return bless $self, $pkg;
}


sub parse {
    my $self = shift;

    if (!open (IN, "< " . $self->{'filename'})) {
        Savefile::log 0, "Failed to open file " . $self->{'filename'} . ": $!";
        return 0;
    }


    my @tokens = ();
    my $line;
    while ($line = <IN>) {
        last if $line =~ /^\s*eyr = { /;     # ignore anything after the country section
        while (length($line)) {
            if ($line =~ /^\s+/) { $line = $'; next;}        # whitespace
            if ($line =~ /^\#.*/) { $line = $'; next;}        #comment

            if ($line =~ /^(^\".*\")\s*$/) { push @tokens, $1; $line = $'; next;}
                #quoted string #1. Matches "Zweite Armeecorps "Neumark" " (nested double-quotes, can only parse by assuming it takes up a single line)

            if ($line =~ /^(\"[^\"]*\")/) { push @tokens, $1; $line = $'; next;}
                #quoted string #2. basic--not nesting

            if ($line =~ /^([0-9+-.]+)/) { push @tokens, $1; $line = $'; next;}  #number
            if ($line =~ /^([A-Za-z0-9_]+|=|\{|\})/) { push @tokens, $1; $line = $'; next;}  #any other token
            else {
                Savefile::log 0, "Parse error: Couldn't get a token out of '$line'";
                $line = '';
            }
        }
    }

    close IN;

    Savefile::log 3, scalar(@tokens) . " tokens parsed";

    while (@tokens) {
        my $header = Savefile::Header->new;
        $self->{'header'} = $header;
        $header->parse(\@tokens) or return _abortParseError(\@tokens);

        my $globaldata = Savefile::Globaldata->new;
        $self->{'globaldata'} = $globaldata;
        $globaldata->parse(\@tokens) or return _abortParseError(\@tokens);

        while (&matches(\@tokens, qw( province = { ))) {
            my $province = Savefile::Province->new;
            $province->parse(\@tokens) or return _abortParseError(\@tokens);
            unless ($province->{water_prov_fl}) {
                $self->addProvince( $province );
            }
        }

        while (&matches(\@tokens, qw( cot = { ))) {
            my $cot = Savefile::CoT->new;
            $cot->parse(\@tokens) or return _abortParseError(\@tokens);
            $self->addCoT( $cot );
        }

        while (&matches(\@tokens, qw( country = { ))) {
            my $ctry = Savefile::Country->new;
            $ctry->parse(\@tokens, $self) or return _abortParseError(\@tokens);
            $self->addCountry( $ctry );

            $self->setProvinceOwners( $ctry );

        }

        last;   # skip anything else in the file
    }



    1;
}


sub _abortParseError {
    my ($tokens) = @_;
    Savefile->warnBadState('??', $tokens);

    my $time = time();
    open (ERRLOG, "> err_$time.log");       ## just ignore errors, instead of exiting
    print ERRLOG join(' ', @$tokens) . "\n";
    close ERRLOG;
    return 0;
}

sub addProvince {
    my ($self, $prov) = @_;
    my $id = $prov->{'id'};
    if (defined $self->{'provinces'}->{$id}) {
        Savefile::log 0, "Warning: Province with ID '$id' already defined.";
        return;
    } else {
        $self->{'provinces'}->{$id} = $prov;
    }
}


sub addCoT {
    my ($self, $cot) = @_;
    $self->{'cots'}->{ $cot->{'location'} } = $cot;
}


sub addCountry {
    my ($self, $ctry) = @_;
    my $tag = $ctry->{'tag'};
    if (defined $self->{'countries'}->{$tag}) {
        Savefile::log 0, "Warning: Country with tag '$tag' already defined.";
        return;
    } else {
        $self->{'countries'}->{$tag} = $ctry;
    }

}

sub getCountry {
    my ($self, $tag) = @_;
    return $self->{'countries'}->{$tag};
}


sub setProvinceOwners {
    my ($self, $country) = @_;
    my $tag = $country->{'tag'};
    foreach ( @{$country->{'ownedprovinces'}}) {
        $self->{'provinces'}->{$_}->{'owner'} = $tag;
    }
}


sub getGameDate {
    my $self = shift;
    return $self->{'globaldata'}->{'startdate'};
}


#--------- Output functions ----------------------------

sub logGlobal {
    @watchesGlobal = @_;
}

sub logCountry {
    @watchesCountry = @_;
}

sub OutputDir {
    $outputDir = shift;

    use File::Path;

    mkpath($outputDir, 1);
}



sub dumpHeader {
    foreach my $watch (@watchesCountry) {
        #if (Savefile->can($watch)) {
            my $filename = "$outputDir/country_$watch.txt";
            Savefile::log 2, "Writing country data to output file \"country_$watch.txt\"";
            open (OUT, ">> $filename") or die $!;
            print OUT join("\t", 'Date     ', 'Year', sort keys %{$dbData->{countries}}), "\n";
            close OUT;
        #}
        #else {
        #    warn "Don't know how to calculate country output $watch\n";;
        #}
    }

    if (@watchesGlobal) {
        my $filename = "$outputDir/globals.txt";
        Savefile::log 2, "Writing global data to file \"globals.txt\"";
        my @tmp;
        foreach my $watch (@watchesGlobal) {
            if (Savefile->can($watch)) {
                push @tmp, Savefile->$watch(undef, 'header');
            }
            else {
                push @tmp, $watch;
            }
            #else {
            #    Savefile::log 0, "Warning: Don't know how to calculate global output $watch";
            #}
        }

        open (OUT, ">> $outputDir/globals.txt") or die $!;
        print OUT join("\t", @tmp), "\n";
        close OUT;
    }

}


sub dumpData {
    my $savedata = shift;

    my $watch;
    foreach $watch (@watchesCountry) {
        #next unless Savefile->can($watch);

        open (OUT, ">> $outputDir/country_$watch.txt") or die $!;
        print OUT $savedata->getGameDate, "\t", $savedata->dumpStartYear;

        if ($savedata->can($watch)) {
            print OUT "\t", join("\t", $savedata->$watch), "\n";
        }
        else {
            foreach my $tag (sort keys %{$Savefile::dbData->{countries}}) {
                my $obj = $savedata->getCountry($tag);
                print OUT "\t";
                if (defined $obj) {
                    if ($obj->can($watch)) {
                        print OUT $obj->$watch();
                    }
                    elsif (defined $obj->{$watch}) {
                        print OUT $obj->{$watch};
                    }
                    else {
                        print OUT '<undef>';
                    }
                }
            }

            print OUT "\n";

        }
        close OUT;
    }

    open (OUT, ">> $outputDir/globals.txt") or die $!;
    my @tmp = ();
    foreach $watch (@watchesGlobal) {
        if ($savedata->can($watch)) {
            push @tmp, $savedata->$watch();
        }
        elsif (defined $savedata->{$watch}) {
            push @tmp, $savedata->{$watch};
        }
        elsif ($savedata->{'header'}->can($watch)) {
            push @tmp, $savedata->{'header'}->$watch();
        }
        elsif (defined $savedata->{'header'}->{$watch}) {
            push @tmp, $savedata->{'header'}->{$watch};
        }
        elsif ($savedata->{'globaldata'}->can($watch)) {
            push @tmp, $savedata->{'globaldata'}->$watch();
        }
        elsif (defined $savedata->{'globaldata'}->{$watch}) {
            push @tmp, $savedata->{'globaldata'}->{$watch};
        }
        else {
            push @tmp, '<undefined>';
        }
    }

    print OUT join("\t", @tmp), "\n";
    close OUT;

}


sub dumpDbData {
    my ($var, @fields) = @_;

    if (!defined $Savefile::dbData->{$var}) {     ## really, this should be verifying the fields are hash types
        Savefile::log 0, "Game data type '$var' is unknown. Available types:\n    "
        . join("\n    ", sort keys %{$Savefile::dbData}) . "\n";

        return;
    }

    open (OUT, "> $outputDir/db_$var.txt") or die $!;
    print OUT join("\t", 'key', @fields),"\n";

    foreach my $key (sort keys %{$Savefile::dbData->{$var}}) {
        my $obj = $Savefile::dbData->{$var}->{$key};
        print OUT $key;
        foreach my $field (@fields) {
            my $value = $obj->{$field};
            print OUT "\t", (defined $value and !ref $value) ? $value : '<undef>';
        }
        print OUT "\n";
    }

    close OUT;

    Savefile::log 2, "Writing DB '$var' data to \"db_$var.txt\"";
}


#--------- Output data types ---------

#------- Global Dump Functions

sub dumpStartYear {
    my $self = shift;
    return (ref $self) ? substr($self->getGameDate, 0, 4) : 'Year';
}


sub dumpTimestampFormat {
    my $self = shift;
    if (!ref $self) {
           return "Saved Time";
       } else {
           my @x = localtime($self->{'timestamp'});
           return sprintf("%04d/%02d/%02d %02d:%02d:%02d", $x[5]+1900, $x[4]+1, $x[3], @x[2,1,0]);
       }
}


# Global, flags for discoveries of goal regions
sub dumpDiscoveredRegions {
    my $self = shift;
    if (!ref $self) {
        return "Discovered: " . join("\t", @DISCOVERIES);
    } else {
        my @ret;
        foreach (@DISCOVERIES) {
            push @ret, $self->{'globaldata'}->{'discovered_regions'}->{$_};
        }
        return join("\t", @ret);
    }
}



# Global, flags for establishments in goal regions
sub dumpEstablishedRegions {
    my $self = shift;
    if (!ref $self) {
        return "Established: " . join("\t", @ESTABLISHMENTS);
    } else {
        my @ret;
        foreach (@ESTABLISHMENTS) {
            push @ret, $self->{'globaldata'}->{'estab_regions'}->{$_};
        }
        return join("\t", @ret);
    }
}


sub dumpEventFlags {
    my $self = shift;
    if (!ref $self) {
        return join("\t", @FLAGS);
    } else {
        my @ret;
        foreach (@FLAGS) {
            push @ret, ($self->{'flags'}->{$_} || '0');
        }
        return join("\t", @ret);
    }
}


sub dumpTotalNativePop {
    my $self = shift;
    if (!ref $self) {
        return 'Native Population';
    } else {
        my $ret = 0;

        foreach my $id ( keys %{$Savefile::dbData->{'provinces'}} ) {
            my $prov = $self->{'provinces'}->{$id} || $Savefile::dbData->{'provinces'}->{$id};
            $ret += ($prov->{'natives'} || 0);
        }

        return $ret;
    }
}


sub dumpNativePopPerContinent {
    my $self = shift;
    if (!ref $self) {
        my @ret;
        foreach (sort @CONTINENTS) {
            push @ret, "Native Pop/$_";
        }
        return join("\t", @ret);
    } else {
        my %ret;

        foreach my $id ( keys %{$Savefile::dbData->{'provinces'}} ) {
            my $prov = $self->{'provinces'}->{$id} || $Savefile::dbData->{'provinces'}->{$id};
            $ret{ $prov->{'continent'} } += ($prov->{'natives'} || 0);
        }

        my @ret;
        foreach (sort @CONTINENTS) {
            push @ret, ($ret{$_} || 0);
        }

        return join("\t", @ret);
    }
}


# Total values from every CoT
sub dumpTotalMerchantIncome {
    my $self = shift;
    if (!ref $self) {
        return "Merchant Income";
    } else {
        my $ret;
        foreach my $cot (values %{$self->{'cots'}}) {   #foreach province location number
            foreach my $levels (values %{$cot->{'merchants'}}) {
                $ret += $levels->[1];
            }
        }

        return $ret;
    }
}

# Total number of traders from every CoT
sub dumpTotalMerchants {
    my $self = shift;
    if (!ref $self) {
        return "Num Merchants";
    } else {
        my $ret;
        foreach my $cot (values %{$self->{'cots'}}) {   #foreach province location number
            foreach my $levels (values %{$cot->{'merchants'}}) {
                $ret += $levels->[0];
            }
        }

        return $ret;
    }
}

# Total number of CoTs
sub dumpNumCoTs {
    my $self = shift;
    if (!ref $self) {
        return "Num CoTs";
    } else {
        return scalar keys %{$self->{'cots'}};
    }
}


#------- Per Country Dump Functions

# Per country, number of CoTs possessed
sub dumpNumCotOwned {
    my $self = shift;
    if (!ref $self) {
        return "{dumpNumCotOwned} Country headings should be generated in dumpHeader";
    } else {
        my %ret;
        foreach (keys %{$self->{'cots'}}) {   #foreach province location number
            my $owner = $self->{'provinces'}->{$_}->{'owner'};
            if (defined $owner) {
                $ret{$owner}++;
            } else {
                Savefile::log 0, "Wow- CoT in unowned province [$_] " . $self->{'provinces'}->{$_}->{'name'};
            }
        }

        my @ret;
        foreach (sort keys %{$Savefile::dbData->{countries}}) {
            push @ret, ( $ret{$_} or 0);
        }

        return (@ret);
    }
}


# Per country, number of CoTs with >=6 merchants
sub dumpNumCotMonopoly {
    my $self = shift;
    if (!ref $self) {
        return "{dumpNumCotMonopoly} Country headings should be generated in dumpHeader";
    } else {
        my %ret;
        foreach my $cot (values %{$self->{'cots'}}) {   #foreach province location number
            while (my ($tag, $levels) = each %{$cot->{'merchants'}}) {
                if ($levels->[0] >= 6) {
                    $ret{$tag}++
                }
            }
        }

        my @ret;
        foreach (sort keys %{$Savefile::dbData->{countries}}) {
            push @ret, ( $ret{$_} or 0);
        }

        return (@ret);
    }
}




# Per country, number of CoTs with >=1 merchant
sub dumpNumCotPresence {
    my $self = shift;
    if (!ref $self) {
        return "{dumpNumCotPresence} Country headings should be generated in dumpHeader";
    } else {
        my %ret;
        foreach my $cot (values %{$self->{'cots'}}) {   #foreach province location number
            while (my ($tag, $levels) = each %{$cot->{'merchants'}}) {
                if ($levels->[0] >= 1) {
                    $ret{$tag}++
                }
            }
        }

        my @ret;
        foreach (sort keys %{$Savefile::dbData->{countries}}) {
            push @ret, ( $ret{$_} or 0);
        }

        return (@ret);
    }
}


# Per country, total number of traders in all CoTs
sub dumpNumMerchants {
    my $self = shift;
    if (!ref $self) {
        return "{dumpNumMerchants} Country headings should be generated in dumpHeader";
    } else {
        my %ret;
        foreach my $cot (values %{$self->{'cots'}}) {   #foreach province location number
            while (my ($tag, $levels) = each %{$cot->{'merchants'}}) {
                $ret{$tag} += $levels->[0];
            }
        }

        my @ret;
        foreach (sort keys %{$Savefile::dbData->{countries}}) {
            push @ret, ( $ret{$_} or 0);
        }

        return (@ret);
    }
}


# Per country, total income from all CoTs
sub dumpMerchantIncome {
    my $self = shift;
    if (!ref $self) {
        return "{dumpMerchantIncome} Country headings should be generated in dumpHeader";
    } else {
        my %ret;
        foreach my $cot (values %{$self->{'cots'}}) {   #foreach province location number
            while (my ($tag, $levels) = each %{$cot->{'merchants'}}) {
                $ret{$tag} += $levels->[1];
            }
        }

        my @ret;
        foreach (sort keys %{$Savefile::dbData->{countries}}) {
            push @ret, ( $ret{$_} or 0);
        }

        return (@ret);
    }
}


# Per country, is there a record for it in the savefile
sub dumpCountryExists {
    my $self = shift;
    if (!ref $self) {
        return "{dumpCountryExists} Country headings should be generated in dumpHeader";
    } else {
        my @ret;
        foreach my $tag (sort keys %{$Savefile::dbData->{countries}}) {
            push @ret, (defined $self->{countries}->{$tag}) ? 'Yes' : '';
        }
        return @ret;
    }
}


# Per country, is there a record for it in the savefile
sub dumpPlayerName {
    my $self = shift;
    if (!ref $self) {
        return "{dumpPlayerName} Country headings should be generated in dumpHeader";
    } else {
        my @ret;
        foreach my $tag (sort keys %{$Savefile::dbData->{countries}}) {
            push @ret, ($self->{header}{playernames}{$tag} || '');
        }
        return @ret;
    }
}




#  ------- UTILITIES ---------
sub balanceParen {
    my ($toks, $pre, $post) = @_;
    my $count = 1;

    while (@$toks) {
        my $cur = shift @$toks;
        if ($cur eq $pre) { ++$count; }
        elsif ($cur eq $post) { --$count; return if $count == 0; }
    }

    Savefile::log 0, "Unmatched '$pre' before end of buffer after '" . join(' ', @$toks[0..10]) . "'";
    return;
}


sub warnBadState {          #($state, $buf);
    my ($class, $state, $toks) = @_;
    Savefile::log 0, "Stuck in object $class, state $state, with buffer '" . join(' ', @$toks[0..24]) . "'";
}


sub matches {            # (\@tokens, @fields)
    my ($toks, @fields) = @_;

    foreach (0 .. $#fields) {
        return 0 if !defined $toks->[$_];
        return 0 if $toks->[$_] !~ /^$fields[$_]$/;
   }

   1;
}



#-------------------------------------
package Savefile::Header;

sub new {
    my $pkg = shift;
    my $self = {
        'gametype' => undef,
        'saved' => undef,
        'free' => undef,
        'optionmode' => undef,
        'set_ai_aggresive' => undef,
        'set_difficulty' => undef,
        'set_gamespeed' => undef,
        'set_fow' => undef,
        'set_conq_capital' => undef,
        'set_missions' => undef,
        'set_basevp' => undef,
        'playernames' => {},
    };

    return bless $self, $pkg;
}


# Global output function: live players from the header block
sub dumpPlayers {
    my $self = shift;
    if (!ref $self) {
        return "Players";
    } else {
        my @ret;
        foreach (sort keys %{$self->{'playernames'}}) {
            push @ret, "$_=\"" . $self->{'playernames'}->{$_} . "\"";
        }
        return join(',', @ret);
    }
}


sub parse {
    my ($obj, $toks) = @_;
    my $state = '<start>';
    my $playerTag = undef;

    while ($state ne '<end>') {
        if (scalar @$toks == 0) {  ## nothing left in the whole file
            $state = '<end>';
            next;
        }

        if ($state eq '<start>') {
            if ( Savefile::matches( $toks, qw(header = { ))) {
                $state = '<data>';
                splice (@$toks, 0, 3);
            } else {
                Savefile::log 0, "Warning: Expected 'header = {', not '" . join(' ', @$toks[0..10]) . "...'";
                return 0;
            }
        }

        elsif ($state eq '<data>') {
            # end of block
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<end>';
                splice (@$toks, 0, 1);
                next;
            }
            # block-valued lines to ignore
            if ( Savefile::matches( $toks, qw( (selectable|id) = { ))) {
                splice (@$toks, 0, 3);
                Savefile::balanceParen( $toks, '{', '}');
                next;
            }
            # single-valued lines to ignore
            if ( Savefile::matches( $toks, qw( (name|tutorial|startyear|endyear|optionfile) = .* ))) {
                splice (@$toks, 0, 3);
                next;
            }
            # single-valued lines to blindly set
            if ( Savefile::matches( $toks, qw( (gametype|saved|free|optionmode|set_ai_aggresive|set_difficulty|set_gamespeed|set_fow|set_conq_capital|set_missions|set_basevp) = .* ))) {
                $obj->{$toks->[0]} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            #anything else must be a country tag
            if ( Savefile::matches( $toks, qw( \w{3} = { ))) {
                $playerTag = $toks->[0];
                $state = '<playername>';
                splice (@$toks, 0, 3);
                next;
            }
            #....................
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;
            }
        }
        elsif ($state eq '<playername>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                $playerTag = undef;
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( playername = .* ))) {
                my $name = $toks->[2];
                splice (@$toks, 0, 3);
                if ($name =~ /^"[^:]*: "$/) {
                    #just an ai player?
                    next;
                }
                if ($name =~ /^\"[^:]*: ([^\"]+)\"$/) {
                    $obj->{'playernames'}->{$playerTag} = $1;
                }
            }
            if ( Savefile::matches( $toks, qw( .* = .* ))) {
                # ignore anything else
                splice (@$toks, 0, 3);
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;
            }
        }
        else {
            Savefile::warnBadState(ref $obj, $state, $toks);
            return 0;
        }
    } #while

    1;
}



#-------------------------------------
package Savefile::Globaldata;

sub new {
    my $pkg = shift;
    my $self = {
        'startdate' => undef,
        'emperor' => undef,
        'discovered_regions' => {},             # ...
        'estab_regions' => {},              # ...
        'flags' => {},
    };

    return bless $self, $pkg;
}

sub parse {
    my ($obj, $toks) = @_;
    my $state = '<start>';

    while ($state ne '<end>') {
        if (scalar @$toks == 0) {  ## nothing left in the whole file
            $state = '<end>';
            next;
        }

        if ($state eq '<start>') {
            if ( Savefile::matches( $toks, qw(globaldata = \{ ))) {
                $state = '<data>';
                splice (@$toks, 0, 3);
            } else {
                Savefile::log 0, "Warning: Expected 'globaldata = {', not '" . join(' ', @$toks[0..10]) . "...'";
                return 0;
            }
        }

        elsif ($state eq '<data>') {
            # startdate
            if ( Savefile::matches( $toks, qw(startdate = \{ year = \d+ month = \w+ day = \d+ \} ))) {
                #Note: If day is '00' change it to '01' (So spreadsheets can handle it)
                $obj->{'startdate'} = sprintf("%04d/%02d/%02d", $toks->[5], $MONTHS{$toks->[8]}, ($toks->[11]||1));
                splice (@$toks, 0, 13);
                next;
            }
            # enddate
            if ( Savefile::matches( $toks, qw(enddate = \{ year = \d+ month = \w+ day = \d+ \} ))) {
                #nothing
                splice (@$toks, 0, 13);
                next;
            }
            #name
            if ( Savefile::matches( $toks, qw(name = \"[^"]*" ))) {
                #nothing
                splice (@$toks, 0, 3);
                next;
            }
            if ( Savefile::matches( $toks, qw(desc = \"[^"]*" ))) {
                #nothing
                splice (@$toks, 0, 3);
                next;
            }
            if ( Savefile::matches( $toks, qw(emperor = \w{3}  ))) {
                $obj->{'emperor'} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            if ( Savefile::matches( $toks, qw( discoveries = { ))) {
                $state = '<discoveries>';
                splice (@$toks, 0, 3);
                next;
            }
            if ( Savefile::matches( $toks, qw( establishments = { ))) {
                $state = '<establishments>';
                splice (@$toks, 0, 3);
                next;
            }
            if ( Savefile::matches( $toks, qw( alliance|war|progress = { ))) {
                splice (@$toks, 0, 3);
                #ignore
                Savefile::balanceParen( $toks, '{', '}');
                next;
            }
            if ( Savefile::matches( $toks, qw( flags = { ))) {
                splice (@$toks, 0, 3);
                $state = '<flags>';
                next;
            }
            if ( Savefile::matches( $toks, qw( dutch = yes|no ))) {
                #ignore
                splice (@$toks, 0, 3);
                next;
            }
            if ( Savefile::matches( $toks, qw( (catholic|orthodox|protestant|sunni|shiite|pagan|reformed|counterreform|buddhism|hinduism|confucianism) = \w{3} ))) {
                #ignore. not sure if all of these are valid, but better to be all-inclusive than to fail a parse
                splice (@$toks, 0, 3);
                next;
            }
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<end>';
                splice (@$toks, 0, 1);
                next;
            }
            #....................
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;
            }


        }
        elsif ($state eq '<discoveries>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( \w+ = yes|no ))) {
                $obj->{'discovered_regions'}->{$toks->[0]} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }
        elsif ($state eq '<establishments>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( \w+ = yes|no ))) {
                $obj->{'estab_regions'}->{$toks->[0]} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }
        elsif ($state eq '<flags>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( \d+ ))) {
                $obj->{'flags'}->{ $FLAGS[$toks->[0]] } = 1;
                splice (@$toks, 0, 1);
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }
        else {
            Savefile::warnBadState(ref $obj, $state, $toks);
            return 0;
        }
    } #while

    1;
}



#-------------------------------------
package Savefile::Province;

sub new {
    my $pkg = shift;
    my $self = {
        'id' => undef,
        'name' => undef,
        'whiteman' => '<undef>',
        'culture' => '<undef>',
        'religion' => '<undef>',
        'income' => undef,
        'manpower' => undef,
        'tax' => undef,
        'natives' => undef,
        'mine' => undef,
        'goods' => undef,
        #ignored: winter, construction, nat'lism, date, control, looted, construction, storm
        #-------
        'city' => undef,      #city data from country section
        'water_prov_fl' => 0,
        #------------
        'area' => undef,
        'region' => undef,
        'continent' => undef,
        #------------
        'owner' => undef,              # The tag of the country that owns this province
    };

    return bless $self, $pkg;
}

sub parse {
    my ($obj, $toks) = @_;
    my $state = '<start>';

    while ($state ne '<end>') {
        if (scalar @$toks == 0) {  ## nothing left in the whole file
            $state = '<end>';
            next;
        }

        if ($state eq '<start>') {
            if ( Savefile::matches( $toks, qw(province = { ))) {
                $state = '<data>';
                splice (@$toks, 0, 3);
            } else {
                Savefile::log 0, "Warning: Expected 'province = {', not '" . join(' ', @$toks[0..10]) . "...'";
                return;
            }
        }

        elsif ($state eq '<data>') {
            # id
            if ( Savefile::matches( $toks, qw(id = \d+ ))) {
                $obj->{'id'} = $toks->[2];
                splice (@$toks, 0, 3);

                #Merge in default data for this province
                my $tprov = &Savefile::getDbProvince( $obj->{'id'} );
                if (defined $tprov) {
                    $obj->{'name'} = $tprov->{'name'};
                    $obj->{'religion'} = $tprov->{'religion'};
                    $obj->{'culture'} = $tprov->{'culture'};
                    $obj->{'manpower'} = $tprov->{'manpower'};
                    $obj->{'income'} = $tprov->{'income'};
                    $obj->{'mine'} = $tprov->{'mine'};
                    $obj->{'natives'} = $tprov->{'natives'};
                    $obj->{'area'} = $tprov->{'area'};
                    $obj->{'region'} = $tprov->{'region'};
                    $obj->{'continent'} = $tprov->{'continent'};
                    $obj->{'goods'} = $tprov->{'goods'};
                    $obj->{'terrain'} = $tprov->{'terrain'};
                }
                else {
                    $obj->{'water_prov_fl'} = 1;
                }

                next;
            }
            # whiteman
            if ( Savefile::matches( $toks, qw(whiteman = yes|no ))) {
                $obj->{'whiteman'} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            # culture
            if ( Savefile::matches( $toks, qw(culture = .* ))) {
                $toks->[2] =~ s/^"(.*)"$/$1/;    # strip off quotes
                $obj->{'culture'} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            # religion
            if ( Savefile::matches( $toks, qw(religion = \w+ ))) {
                $obj->{'religion'} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            # winter/looted/storm/province_revoltrisk = xxx (ignore)
            if ( Savefile::matches( $toks, qw( (winter|looted|storm|province_revoltrisk) = .* ))) {
                #ignore
                splice (@$toks, 0, 3);
                next;
            }
            # natives/mine/income/manpower/tax/goods/terrain = number. So far, I have found only manpower with values <0
            if ( Savefile::matches( $toks, qw( (natives|mine|income|manpower|tax|goods|terrain) = [-\d]+ ))) {
                $obj->{$toks->[0]} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            # construction, etc (ignore)
            if ( Savefile::matches( $toks, qw(construction|nationalism|date|control = { ))) {
                splice (@$toks, 0, 3);
                #ignore
                Savefile::balanceParen( $toks, '{', '}');
                next;
            }
            if ( Savefile::matches( $toks, qw( } ))) {
            #if ($buf =~ /^\}/s) {
                $state = '<end>';
                splice (@$toks, 0, 1);
                next;
            }
            #....................
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }


        }
        else {
            Savefile::warnBadState(ref $obj, $state, $toks);
            return 0;
        }

    } #while

    1;
}

#-------------------------------------
package Savefile::CoT;

sub new {
    my $pkg = shift;
    my $self = {
        'location' => undef,
        'merchants' => {},
        #ignored: extra
    };

    return bless $self, $pkg;
}


sub parse {
    my ($obj, $toks) = @_;
    my $state = '<start>';

    while ($state ne '<end>') {
        if (scalar @$toks == 0) {  ## nothing left in the whole file
            $state = '<end>';
            next;
        }

        if ($state eq '<start>') {
            if ( Savefile::matches( $toks, qw(cot = { ))) {
                $state = '<data>';
                splice (@$toks, 0, 3);
            } else {
                Savefile::log 0, "Warning: Expected 'cot = {', not '" . join(' ', @$toks[0..10]) . "...'";
                return 0;
            }
        }

        elsif ($state eq '<data>') {
            # location
            if ( Savefile::matches( $toks, qw( location = \d+ ))) {
                $obj->{'location'} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }

            if ( Savefile::matches( $toks, qw( merchant = { ))) {
                $state = '<merchant>';
                splice (@$toks, 0, 3);
                next;
            }

            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<end>';
                splice (@$toks, 0, 1);
                next;
            }
            #....................
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;
            }


        }
        elsif ($state eq '<merchant>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( tag = \w{3} level = .* value = .* } ))) {
                unless ($toks->[5]==0) {
                    $obj->{'merchants'}->{$toks->[2]} = [ $toks->[5], $toks->[8] ];
                }
                $state = '<data>';
                splice (@$toks, 0, 10);
                next;
            }
            if ( Savefile::matches( $toks, qw( tag = \w{3} level = .* value = .*  extra = { ))) {
                unless ($toks->[5]==0) {
                    $obj->{'merchants'}->{$toks->[2]} = [ $toks->[5], $toks->[8] ];
                }
                splice (@$toks, 0, 12);
                Savefile::balanceParen( $toks, '{', '}');
                next;
            }
            if ( Savefile::matches( $toks, qw( extra = { ))) {
                splice (@$toks, 0, 3);
                Savefile::balanceParen( $toks, '{', '}');
                next;
            }
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }

            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }

        else {
            Savefile::warnBadState(ref $obj, $state, $toks);
            return 0;

        }
    } #while

    1;

}


#-------------------------------------
package Savefile::Country;



sub new {
    my $pkg = shift;
    my $self = {
        'tag' => undef,
        'policies' => [],
        'cultures' => [],
        'religion' => undef,
        'hist_monarch' => [],    #ADM DIP MIL
        'badboy' => undef,
        'whiteman' => undef,
        'lastmonthincome' => undef,
        'lastmonthexpense' => undef,
        'lastyearincome' => 0,
        'lastyearexpense' => 0,
        #'basevp' => [],
        'vp' => undef,
        'atwar_fl' => undef,
        'treasury' => undef,
        'income' => undef,
        'inflation' => undef,
        'manpower' => undef,
        'navalpercentage ' => undef,
        'landpercentage' => undef,
        'tepercentage' => undef,
        'pepercentage' => undef,
        'stabilitypercentage' => undef,
        'treasurypercentage' => undef,
        'tax' => undef,
        'num_known_provs' => 0,      # calc'd from the knownprovinces list
        'ownedprovinces' => [],
        'num_warships' => 0,
        'num_galleys' => 0,
        'num_transports' => 0,
        'num_inf' => 0,
        'num_cav' => 0,
        'num_art' => 0,
        'tech_levels' => [],
        'tech_current' => [],
        'tech_group' => undef,
        'electors' => 0,
    };


        # what's this?? $merchants => {},


        #ignored: colonialnation, major, bank, stock, beuraucracy, revolt, revoltrisk, revoltriskdate,
        #         landmaintenance, navalmaintenance, diplomacy, tax,
        #         missionpercentage, missiontype, missionareaname, missionname, missionarea,
        #         controlledprovinces, nationalprovinces

    return bless $self, $pkg;
}


sub countOwnedProvinces {
    my $self = shift;
    if (!ref $self) {
        return 'ownedprovinces';
    } else {
        return 1+$#{$self->{'ownedprovinces'}};
    }
}

sub countKnownProvinces {
    my $self = shift;
    if (!ref $self) {
        return 'knownprovinces';
    } else {
        return $self->{'num_known_provs'};
    }
}


sub DiploString {
    my $self = shift;
    if (!ref $self) {
        return 'DP Settings';
    } else {
        return join("-", @{$self->{'policies'}});
    }
}



sub totalArmy {
    my $self = shift;
    if (!ref $self) {
        return 'Army Strength';
    } else {
        return $self->{num_inf} + $self->{num_cav} + $self->{num_art};
    }
}



sub totalNavy {
    my $self = shift;
    if (!ref $self) {
        return 'Navy Strength';
    } else {
        return $self->{num_warships} + $self->{num_galleys} + $self->{num_transports};
    }
}




sub parse {
    my ($obj, $toks, $savedata) = @_;
    my $state = '<start>';

    while ($state ne '<end>') {
        if (scalar @$toks == 0) {  ## nothing left in the whole file
            $state = '<end>';
            next;
        }

        if ($state eq '<start>') {
            if ( Savefile::matches( $toks, qw(country = { ))) {
                $state = '<data>';
                splice (@$toks, 0, 3);
            } else {
                Savefile::log 0, "Warning: Expected 'country = {', not '" . join(' ', @$toks[0..10]) . "...'";
                return;
            }
        }

        elsif ($state eq '<data>') {
            # id
            if ( Savefile::matches( $toks, qw(tag = \w{3} ))) {
                $obj->{'tag'} = $toks->[2];
                splice (@$toks, 0, 3);

                #Merge in default data for this country
                my $tctry = &Savefile::getDbCountry( $obj->{'tag'} );
                if (defined $tctry) {
                    $obj->{'tech_group'} = $tctry->{'tech_group'};
                    $obj->{'policies'} = $tctry->{'policies'};
                    $obj->{'electors'} = $tctry->{'electors'};
                }
                else {
                    Savefile::log 0, "Warning: Savefile country tag $obj->{'tag'} is not defined in country.csv\n";
                }

                next;
            }

            ## New for 1.06 - The ai line went from single-valued to a block. Check for a block to ignore a 1.06 file.
            ## If fails, then check for a single-value ai=.*, which wh\ould <= 1.05
            if ( Savefile::matches( $toks, qw( ai = { ))) {
                splice (@$toks, 0, 3);
                #ignore
                Savefile::balanceParen( $toks, '{', '}');
                next;
            }

            ## New for 1.06 - We're ignoring these single-values currently
            if ( Savefile::matches( $toks, qw( (warexhaustion|estimation) = .* ))) {
                splice (@$toks, 0, 3);
                next;
            }

            ## We're ignoring these currently (note the 1.05 ai=.* syntax)
            if ( Savefile::matches( $toks, qw( (ai|colonialattempts|colonialnation|major|bank|stock|cancelledloans|extendedloans|beuraucracy|revolt|revoltrisk|landmaintenance|navalmaintenance|missionpercentage|missiontype|missionareaname|missionname|loansize) = .* ))) {
                splice (@$toks, 0, 3);
                next;
            }

            ## We're blindly setting these
            if ( Savefile::matches( $toks, qw( (badboy|whiteman|treasury|income|inflation|manpower|navalpercentage|landpercentage|tepercentage|pepercentage|stabilitypercentage|treasurypercentage|tax|colonists|diplomats|merchants|missionaries|dynamism) = .* ))) {
                $obj->{$toks->[0]} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }

            # DP settings
            if ( Savefile::matches( $toks, qw( policy = { ))) {
                splice (@$toks, 0, 3);
                $state = '<policy>';
                next;
            }

            # culture
            if ( Savefile::matches( $toks, qw(culture = { ))) {
                $state = '<culture>';
                splice (@$toks, 0, 3);
                next;
            }

            # ledger
            if ( Savefile::matches( $toks, qw(ledger = { ))) {
                $state = '<ledger>';
                splice (@$toks, 0, 3);
                next;
            }

            # basevp, lastbankruptcy, war, lastpeace, ... (ignore entire block)
            if ( Savefile::matches( $toks, qw( (basevp|lastbankruptcy|war|lastpeace|combatlosses|attritionlosses|disbandlosses|revoltriskdate|historicalmonarch|diplomacy|changedreligion|missionarea|controlledprovinces|nationalprovinces|city|tradingpost|monarchtable|leadertable|mission|diplomaticdates|wardates|loan|monarch) = { ))) {
                splice (@$toks, 0, 3);
                #ignore
                Savefile::balanceParen( $toks, '{', '}');
                next;
            }

            if ( Savefile::matches( $toks, qw( vp = .* ))) {
                ## It looks like for human players, the vp line is defined twice. We will take the first vp as the
                ## value to store, and if the second value is !=0, this will signal a human player
                if (defined $obj->{'vp'}) {
                    if ($toks->[2] != 0 and !defined $savedata->{header}{playernames}{ $obj->{tag} }) {
                        $savedata->{header}{playernames}{ $obj->{tag} } = 'unknown';
                    }
                } else {
                    $obj->{$toks->[0]} = $toks->[2] unless defined $obj->{$toks->[0]};   # There are duplicate vp values. The 2nd are mostly 0, except for the player
                }
                splice (@$toks, 0, 3);
                next;
            }

            # religion
            if ( Savefile::matches( $toks, qw(religion = { ))) {
                $state = '<religion>';
                splice (@$toks, 0, 3);
                next;
            }

            # knownprovinces
            if ( Savefile::matches( $toks, qw(knownprovinces = { ))) {
                $state = '<knownprovinces>';
                splice (@$toks, 0, 3);
                next;
            }

            # ownedprovinces
            if ( Savefile::matches( $toks, qw(ownedprovinces = { ))) {
                $state = '<ownedprovinces>';
                splice (@$toks, 0, 3);
                next;
            }

            # technology
            if ( Savefile::matches( $toks, qw(technology = { ))) {
                $state = '<technology>';
                splice (@$toks, 0, 3);
                next;
            }

            # land/naval units
            if ( Savefile::matches( $toks, qw( (landunit|navalunit) = { ))) {
                $state = '<military>';
                splice (@$toks, 0, 3);
                next;
            }

            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<end>';
                splice (@$toks, 0, 1);
                next;
            }
            #....................
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }


        }
        elsif ($state eq '<policy>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( date = { year = .* month = .* day = .* } ))) {
                #ignore
                splice (@$toks, 0, 13);
                next;
            }
            if ( Savefile::matches( $toks, qw( aristocracy|centralization|innovative|mercantilism|offensive|land|quality|serfdom = \d+ ))) {
                if (defined $POLICY_FIELDS{$toks->[0]}) {   # If this dp setting is known, get its position within the array
                    $obj->{'policies'}->[ $POLICY_FIELDS{$toks->[0]} ] = $toks->[2];
                }
                else {
                    Savefile::log 0, "Warning: Unknown DP category '" . $toks->[0] . "'\n";
                }
                splice (@$toks, 0, 3);
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }

        elsif ($state eq '<culture>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( type = .* ))) {
                $toks->[2] =~ s/^"(.*)"$/$1/;    # strip off quotes
                push @{$obj->{'cultures'}}, $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }

        elsif ($state eq '<ledger>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( lastyearincome = { .* .* .* .* .* .* .* .* .* .* .* .* } ))) {
                # There are 12 fields between the brackets, which will be summed
                my $field = $toks->[0];
                splice (@$toks, 0, 3);
                foreach my $i (0 .. 11) {
                    $obj->{$field} += $toks->[0];
                    splice (@$toks, 0, 1);
                }
                splice (@$toks, 0, 1);     # shift the closing '}'
                next;
            }

            if ( Savefile::matches( $toks, qw( lastyearexpense = { .* .* .* .* .* .* .* .* .* .* .* .* .* .* .* .* } ))) {
                # There are 16 fields between the brackets, which will be summed
                my $field = $toks->[0];
                splice (@$toks, 0, 3);
                foreach my $i (0 .. 15) {
                    $obj->{$field} += $toks->[0];
                    splice (@$toks, 0, 1);
                }
                splice (@$toks, 0, 1);     # shift the closing '}'
                next;
            }

            if ( Savefile::matches( $toks, qw( (income|expense|thismonthincome|thismonthexpense) = { ))) {
                # ignoring for now
                splice (@$toks, 0, 3);
                Savefile::balanceParen( $toks, '{', '}');
                next;
            }

            if ( Savefile::matches( $toks, qw( lastmonthincome|lastmonthexpense = .* ))) {
                # blindly set
                $obj->{$toks->[0]} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }

        elsif ($state eq '<religion>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( type = \w+ ))) {
                $obj->{'religion'} = $toks->[2];
                splice (@$toks, 0, 3);
                ## Skip anything else in the block
                Savefile::balanceParen( $toks, '{', '}');
                $state = '<data>';
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }


        elsif ($state eq '<military>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( (inf|cav|art|warships|galleys|transports) = [.\d]+ ))) {
                $obj->{"num_" . $toks->[0]} += $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            if ( Savefile::matches( $toks, qw( \w+ = { ))) {
                # Skip anything else that's blocky
                splice (@$toks, 0, 3);
                Savefile::balanceParen( $toks, '{', '}');
                next;
            }
            if ( Savefile::matches( $toks, qw( \w+ = .* ))) {
                # Skip anything else not blocky
                splice (@$toks, 0, 3);
                next;
            }
            do {

                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }
        elsif ($state eq '<knownprovinces>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( \d+ ))) {
                ++$obj->{'num_known_provs'};
                splice (@$toks, 0, 1);
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }

        elsif ($state eq '<ownedprovinces>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( \d+ ))) {
                push @{$obj->{'ownedprovinces'}}, $toks->[0];
                splice (@$toks, 0, 1);
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }

        elsif ($state eq '<technology>') {
            if ( Savefile::matches( $toks, qw( } ))) {
                $state = '<data>';
                splice (@$toks, 0, 1);
                next;
            }
            if ( Savefile::matches( $toks, qw( .* = { date = { year = \d+ month = \w+ day = \d+ } level = [-\d]+ value = \d+ } ))) {
                if (defined $TECH_FIELDS{$toks->[0]}) {   # If this dp setting is known, get its position within the array
                    $obj->{'tech_levels'}->[ $TECH_FIELDS{$toks->[0]} ] = $toks->[18];
                    $obj->{'tech_current'}->[ $TECH_FIELDS{$toks->[0]} ] = $toks->[21];
                }
                else {
                    Savefile::log 0, "Warning: Unknown Tech category '" . $toks->[0] . "'\n";
                }
                splice (@$toks, 0, 23);
                next;
            }
            if ( Savefile::matches( $toks, qw( group = .* ))) {
                $obj->{'tech_group'} = $toks->[2];
                splice (@$toks, 0, 3);
                next;
            }
            do {
                Savefile::warnBadState(ref $obj, $state, $toks);
                return 0;

            }
        }


        else {
            Savefile::warnBadState(ref $obj, $state, $toks);
            return 0;

        }

    } #while

    1;
}


#-------------------------------------
package Savefile::City;
# includes colonies (when pop < 700)



sub new {
    my $pkg = shift;
    my $self = {
        'id' => undef,                 # location
        'name' => undef,
        'population' => undef,
        'shipyard_fl' => undef,
        'bailiff_fl' => undef,
        'fortress' => undef,
        'capital_fl' => undef,
        'manufactory_fl' => undef,
    };

    return bless $self, $pkg;
}


#-------------------------------------

package Savefile::TradingPost;

sub new {
    my $pkg = shift;
    my $self = {
        'id' => undef,           #location
        'name' => undef,
        'level' => undef,
    };

    return bless $self, $pkg;
}


#-------------------------------------

1;
